;#########################################################################;
;                                                                         ;
;  File:        2D_FFT.ncl                                                ;
;                                                                         ;
;  Author:      Verica Savic-Jovcic                                       ;
;                                                                         ;
;  Date:        March 10 1006                                             ;
;                                                                         ;
;                                                                         ;
;                                                                         ;
;  This file contains functions for calculating the two-dimensional       ;
;      Fourier analysis, as well as the functions for calculating the     ;
;      spectra as a function of a magnitude of the 2D wave number.        ;
;                                                                         ;
;                                                                         ;
;  The functions for 2D fft require input of a two-dimensional variable   ;
;      with the same number of points in both dimensions. The number of   ;
;      points in each dimension should be a power of 2. Furthermore, it   ;
;      requiers as imput several more variables: the number of points,    ;
;      two-dimensional wave number, and an indicator of the choice of     ;
;      output.                                                            ;
;                                                                         ;
;  As an output the functions provide three possible results:             ;
;      1) 2D amplitude if an indicator inr is equal to 0,                 ;
;      2) 2D energy spectra if inr is 1, and                              ;
;      3) 2D power spectra if inr is 2.                                   ;
;                                                                         ;
;  There are two functions: "two_D_fft_xy" and "two_D_fft_yx". The first  ;
;      one calculates 1D FFT in x direction and then 1D FFT in y          ;
;      direction, while the latter does the opposite. However, the final  ;
;      result of the power spectra should be same for both.               ;
;                                                                         ;
;                                                                         ;
;  The functions for calculating spectra require an input of the 2-D wave ;
;      amplitude, 2-D wave energy, 2-D wave number, number of points in   ;
;      the original data series, 1-D wave numbers -- one to define the    ;
;      limits for the averaging and one to represent the apparent 1D wave ;
;      number -- and a flag to define the output -- 0 returnes the 1-D    ;
;      wave amplitude, 1 returns the 1-D wave energy and 2 returns the    ;
;      1-D wave spectra (energy multiplied by the wave number).           ;
;                                                                         ;
;  These functions cut off the unrepresented large wave numbers.          ;
;                                                                         ;
;  There are two functions: "two_to_one_D_spectra_xy" and                 ;
;      "two_to_one_D_spectra_yx". The first uses the results of the       ;
;      "two_D_fft_xy", while the sond one uses the results of the         ;
;      "two_D_fft_yx". We are aware of no need for this distinction, but  ;
;      we keep it for the sake of completeness as this code is still in   ;
;      the development phase.                                             ;
;                                                                         ;
;                                                                         ;
;                                                                         ;
;  In addition to the above functions, there are two more that provide    ;
;      the support for them. In particular, "two_D_frequency" formulates  ;
;      a 2-D wave numbers and "one_D_frequency" calculates the magnitudes ;
;      of the 2-D wave numbers and organizes them in an 1-D wave number.  ;
;                                                                         ;
;                                                                         ;
;                                                                         ;
;#########################################################################;



;---------------------------------------------------------------------------;

; function two_D_fft_xy does the analysis first in x and then in y direction:

undef("two_D_fft_xy")

function two_D_fft_xy(inp:float,x:float,y:float,cff2D:float, \
            Np:integer,inr:integer)

begin

; input

; constants
  Nph = floattointeger(Np/2.)
  Npm1 = Np-1
  Nphm1 = Nph-1

; data - dimensions
  ReIm = (/0,1/)
  ReIm!0 = "ReIm"
  ReIm&ReIm = ReIm

  cffy = cff2D&cffy
  cffx = cff2D&cffx

; new variables
  dmyxy = cff2D
  dmyxy = cff2D@_FillValue

  cfx = new((/2,Np,Nph/),float)
  cfx!0 = "ReIm"
  cfx&ReIm = ReIm
  cfx!1 = "y"
  cfx&y = y
  cfx!2 = "cffx"
  cfx&cffx = cffx

  dmy = new((/2,Nph,Nph/),float)
  dmy!0 = "ReIm"
  dmy&ReIm = ReIm
  dmy!1 = "cffy"
  dmy&cffy = cffy
  dmy!2 = "cffx"
  dmy&cffx = cffx
  cfxRy = dmy
  cfxIy = dmy

  amplxy = dmyxy
  amplxy@long_name = "amplitude"
  enxy = dmyxy
  enxy@long_name = "energy spectra"
  spxy = dmyxy
  spxy@long_name = "power spectra"

; analize data
  do i = 0, Npm1
    cfx(:,i,:) = (/ezfftf(inp(i,:))/)  ; do fft in x direction
  end do

  do i = 0, Nphm1
    ; do fft of cfx in y direction
    cfxRy(:,:,i) = (/ezfftf(cfx(0,:,i))/)  ; fft of the real part of cfx
    cfxIy(:,:,i) = (/ezfftf(cfx(1,:,i))/)  ; fft of the imaginary part of cfx
    amplxy(:,i) = (/sqrt((cfxRy(0,:,i)-cfxIy(1,:,i))^2 + \
                         (cfxRy(1,:,i)+cfxIy(0,:,i))^2)/)
  end do
  ; spectra 
  amplxy = (/mask(amplxy,(cff2D.le.cffx(Nph-1)),True)/)
  enxy = (/amplxy^2/)
  spxy = (/enxy*cff2D/)

; return results
  if (inr .eq. 0) then
    return(amplxy)
  end if
  if (inr .eq. 1) then
    return(enxy)
  end if
  if (inr .eq. 2) then
    return(spxy)
  end if

end


;---------------------------------------------------------------------------;

; function two_D_fft_yx does the analysis first in y and then in x direction:

undef("two_D_fft_yx")

function two_D_fft_yx(inp:float,x:float,y:float,cff2D:float, \
            Np:integer,inr:integer)

begin

; constants
  Nph = floattointeger(Np/2.)
  Npm1 = Np-1
  Nphm1 = Nph-1

; data - dimensions
  ReIm = (/0,1/)
  ReIm!0 = "ReIm"
  ReIm&ReIm = ReIm

  cffy = cff2D&cffy
  cffx = cff2D&cffx

; new variables
  dmyxy = cff2D
  dmyxy = cff2D@_FillValue

  cfy = new((/2,Nph,Np/),float)
  cfy!0 = "ReIm"
  cfy&ReIm = ReIm
  cfy!1 = "cff"
  cfy&cff = cff
  cfy!2 = "x"
  cfy&x = x

  dmy = new((/2,Nph,Nph/),float)
  dmy!0 = "ReIm"
  dmy&ReIm = ReIm
  dmy!1 = "cffy"
  dmy&cffy = cffy
  dmy!2 = "cffx"
  dmy&cffx = cffx
  cfyRx = dmy
  cfyIx = dmy

  amplyx = dmyyx
  amplyx@long_name = "amplitude"
  enyx = dmyyx
  enyx@long_name = "energy spectra"
  spyx = dmyyx
  spyx@long_name = "power spectra"

; analize data
  do i = 0, Npm1
    cfy(:,:,i) = (/ezfftf(inp(:,i))/)  ; do fft in y direction
  end do

  do i = 0, Nphm1
    ; do fft of cfy in x direction  
    cfyRx(:,i,:) = (/ezfftf(cfy(0,i,:))/)  ; fft of the real part of cfy
    cfyIx(:,i,:) = (/ezfftf(cfy(1,i,:))/)  ; fft of the imaginary part of cfy
    amplyx(i,:) = (/sqrt((cfyRx(0,i,:)-cfyIx(1,i,:))^2 + \
                         (cfyRx(1,i,:)+cfyIx(0,i,:))^2)/)
  end do
  ; spectra 
  amplyx = (/mask(amplyx,(cff2D.le.cffx(Nph-1)),True)/)
  enyx = (/amplyx^2/)
  spyx = (/enyx*cff2D/)

; return results
  if (inr .eq. 0) then
    return(amplyx)
  end if
  if (inr .eq. 1) then
    return(enyx)
  end if
  if (inr .eq. 2) then
    return(spyx)
  end if

end


;---------------------------------------------------------------------------;

; function two_D_frqnc_xy calculates 2D frequency corresponding to 2D spectra

undef("two_D_frqnc_xy")

function two_D_frqnc_xy(Np:integer)

begin

; constants
  Nph = floattointeger(Np/2.)
  Npm1 = Np-1
  Nphm1 = Nph-1

; data - dimensions
  cff = ispan(1,Nph,1)*1.
  cff!0 = "cff"
  cff&cff = cff

  cffx = (/cff/)
  cffx!0 = "cffx"
  cffx&cffx = cffx

  cffy = (/cff/)
  cffy!0 = "cffy"
  cffy&cffy = cffy

  cff2D = new((/Nph,Nph/),float)
  cff2D!0 = "cffx"
  cff2D&cffx = cffx
  cff2D!1 = "cffy"
  cff2D&cffy = cffy
  do i = 0, Nphm1
    do j = 0, Nphm1
      cff2D(i,j) = (/sqrt(cffy(j)^2+cffx(i)^2)/)
    end do
  end do

  cff2D = (/mask(cff2D,(cff2D.le.cffx(Nph-1)),True)/)

  return(cff2D)

end


;---------------------------------------------------------------------------;

; function two_D_frqnc_yx calculates 2D frequency corresponding to 2D spectra

undef("two_D_frqnc_yx")

function two_D_frqnc_yx(Np:integer)

begin

; constants
  Nph = floattointeger(Np/2.)
  Npm1 = Np-1
  Nphm1 = Nph-1

; data - dimensions
  cff = ispan(1,Nph,1)*1.
  cff!0 = "cff"
  cff&cff = cff

  cffx = (/cff/)
  cffx!0 = "cffx"
  cffx&cffx = cffx

  cffy = (/cff/)
  cffy!0 = "cffy"
  cffy&cffy = cffy

  cff2D = new((/Nph,Nph/),float)
  cff2D!0 = "cffy"
  cff2D&cffy = cffy
  cff2D!1 = "cffx"
  cff2D&cffx = cffx
  do j = 0, Nphm1
    do i = 0, Nphm1
      cff2D(j,i) = (/sqrt(cffy(j)^2+cffx(i)^2)/)
    end do
  end do

  cff2D = (/mask(cff2D,(cff2D.le.cffx(Nph-1)),True)/)

  return(cff2D)

end


;---------------------------------------------------------------------------;

; function  one_D_frequency calculates 1D frequency corresponding to 2D spectra
;  results:

undef("one_D_frequency")

function one_D_frequency(Nw:integer,wiw:integer,inr:integer)

begin

  Nwm2 = Nw-2

  k = ispan(wiw,wiw*Nw,wiw)
  k@long_name = "wave number"
  k!0 = "k"
  k&k = k

  wn = new(Nw,float)
  do l = 0, Nwm2
    wn(l) = (/avg(k(l:l+1))/)
  end do
  wn(Nw-1) = k(Nw-1) + wn(Nwm2)
  wn@long_name = "mean wave number"
  wn!0 = "wn"
  wn&wn = wn

  if (inr .eq. 0) then
    return(k)
  end if
  if (inr .eq. 1) then
    return(wn)
  end if

end

;---------------------------------------------------------------------------;

; function two_to_one_D_spectra_xy calculates 1D spectra out of xy 2D fft
;  results:

undef("two_to_one_D_spectra_xy")

function two_to_one_D_spectra_xy(amxy:float,enxy:float,cff2D:float, \
            Np:integer,k:integer,wn:float,inr:integer)

begin

; constants
  Nph = floattointeger(Np/2.)
  Nphm1 = Nph-1
  Nw = dimsizes(k)
  Nwm1 = Nw-1
  Nwm2 = Nw-2

; new variables
  dmy = new(Nw,float,-999.)
  dmy!0 = "wn"
  dmy&wn = wn
  dmy = 0.

  Axy = dmy
  Axy@long_name = "wave amplitude"
  Exy = dmy
  Exy@long_name = "wave energy"
  Sxy = dmy
  Sxy@long_name = "wave power"

; do analysis

  do l = 0, Nwm2
    Axy(l) = Axy(l) + (/avg(mask(log(amxy),(cff2D.lt.k(l+1) \
             .and. cff2D.ge.k(l)),True))/)
    Exy(l) = Exy(l) + (/avg(mask(log(enxy),(cff2D.lt.k(l+1) \
             .and. cff2D.ge.k(l)),True))/)
  end do
  Axy(Nwm1) = -999.
  Exy(Nwm1) = -999.
  Axy = (/exp(Axy)/)
  Exy = (/exp(Exy)/)
  Sxy = (/Exy*wn/)

; return results
  if (inr .eq. 0) then
    return(Axy)
  end if
  if (inr .eq. 1) then
    return(Exy)
  end if
  if (inr .eq. 2) then
    return(Sxy)
  end if

end


;---------------------------------------------------------------------------;

; function two_to_one_D_spectra_yx calculates 1D spectra out of yx 2D fft
;  results:

undef("two_to_one_D_spectra_yx")

function two_to_one_D_spectra_yx(amyx:float,enyx:float,cff2D:float, \
            Np:integer,k:integer,wn:float,inr:integer)

begin

; constants
  Nph = floattointeger(Np/2.)
  Nphm1 = Nph-1
  Nw = dimsizes(k)
  Nwm1 = Nw-1
  Nwm2 = Nw-2

; new variables
  dmy = new(Nw,float)
  dmy!0 = "wn"
  dmy&wn = wn
  dmy = 0.

  Ayx = dmy
  Ayx@long_name = "wave amplitude"
  Eyx = dmy
  Eyx@long_name = "wave energy"
  Syx = dmy
  Syx@long_name = "wave power"
  nmb = dmy

; do analysis
  do l = 0, Nwm2
    Ayx(l) = Ayx(l) + (/avg(mask(log(amyx),(cff2D.lt.k(l+1) \
             .and. cff2D.ge.k(l)),True))/)
    Eyx(l) = Eyx(l) + (/avg(mask(log(enyx),(cff2D.lt.k(l+1) \
             .and. cff2D.ge.k(l)),True))/)
  end do
  Ayx(Nwm1) = -999.
  Eyx(Nwm1) = -999.
  Ayx = (/exp(Ayx)/)
  Eyx = (/exp(Eyx)/)
  Syx = (/Eyx*k/)

; return results
  if (inr .eq. 0) then
    return(Ayx)
  end if
  if (inr .eq. 1) then
    return(Eyx)
  end if
  if (inr .eq. 2) then
    return(Syx)
  end if

end


;---------------------------------------------------------------------------;
