;;;
; From Dennis Shea's posting to NCL user forum
;;;

load "$NCARG_ROOT/lib/ncarg/nclscripts/csm/gsn_code.ncl"
load "$NCARG_ROOT/lib/ncarg/nclscripts/csm/gsn_csm.ncl"
load "$NCARG_ROOT/lib/ncarg/nclscripts/csm/contributed.ncl"
load "$NCARG_ROOT/lib/ncarg/nclscripts/csm/shea_util.ncl"

load "$./fft_tools.ncl"
external FFT2D "$./spectra2d.so"


begin



	;----------------------------------------
	; INPUT
	;               ..........................

	vnam ="olr"  		; Name of variable
	dir = getenv("HOME")
	dir = dir+"/DTMI2/OLR"
	infil = dir+"/"+vnam+"_higc.nc"
	f     = addfile(infil,"r")
	nlen  = 2650          	; No of time steps
	;			; to use
	;----------------------------------------




	;----------------------------------------
	; OUTPUT
	;               ..........................
	
	outfile = "fft_"+vnam+"40-220e_djf.nc"  ; name of output file
	LATAVG=False             ; Do you want the output
	;		  	; written out for all latitudes
	;			; if so, set LATAVG = False
	;
	SYMCALC=False		; Do you want to calculate
	;			; symmetric/asymmetric components
	;			; of the variable about the equator?
	;			; If yes, set to True
	;
	freq_units = "cycles per day"
	;			; set the units for frequency
        wv1=-100		; largest westward wavenumber to write out
        wv2=99			; largest eastward wavenumber to write out
        fr1= 1.0/512.0          ; lowest frequency to write out
        fr2= 1.0/2.0            ; highest frequency to write out
	;----------------------------------------



	;----------------------------------------
	; Parameters for space-time FFT
	;               ..........................

	dt     = 1.     	; sampling interval
	minlat = -15.0          ; range of latitudes -
	maxlat = 15.0		; to consider
	minlon = 40   		; range of latitudes -
	maxlon = 212.5		; to consider
	lsector = 512	 	; length of a time sector
	;		        ; given as a power of 2 for efficiency
	;		        ; prime period is thus 512 days
	; data will be		; and the next few periods are 
	; split into sectors	; 256,171,128,102,85,73,64
 	; of length lsector	; this choice makes sure that i cover
	; the computed spectrum	; the lower frequency flank of intraseasonal
	; will then be averaged	; variability; 
	;----------------------------------------


	;----------------------------------------
	; Read data and its attributes
	;               ..........................

	print("Reading Data")
	varx = f->$vnam$(:nlen-1,{minlat:maxlat},{minlon:maxlon})
	
	;..............
	; Apply Seasonal mask
        print("Calculate mask")
        itest=mask_other_seasons(varx(:,0,0),"djf")
        print("Apply mask")
        varx=varx*conform(varx,itest,0)
	;.............

	lat   = varx&lat
	nlat  = dimsizes(lat)
	lon   = varx&lon
	nlon  = dimsizes(lon)
	print(nlon)
	time  = varx&time
	ntime = dimsizes(time)
	nrepeat=ntime/lsector	; no of time loops
	;---------------------------------------



	;---------------------------------------
	; Check if we want to do Symmetric/Asymmetric calculations
	; and do the necessary
	;               ..........................
	
	if (SYMCALC) then

	  if (.not.isodd(nlat) ) then
  	    print("Require odd number of latitudes")
  	    print("Will not perform this calculation")
	    break

	  else
	    print("SYMMETRIC/ASYMMETRIC CALCULATION")
  	    lat1 = nlat/2 + 1      ; nlat is odd
	    data=varx

	; antisymmetric part is stored in northern hemisphere
	; OLRA(lat) = (OLR(lat)-OLR(-lat))/2.

	    lat2 = nlat-1
	    do n = lat1, lat2
  	    data(:,n,:) = (varx(time|:,lat|n,lon|:) - \ 
		varx(time|:,lat|lat2-n,lon|:))/2.
	    end do

	; symmetric part is stored in southern hemisphere
	; OLRS(lat) = (OLR(lat)+OLR(-lat))/2.

	    do n = lat1, lat2
  	    data(:,lat2-n,:) = (varx(time|:,lat|n,lon|:) + \
                         varx(time|:,lat|lat2-n,lon|:))/2.
	    end do

	   varx=data
	  end if
	end if   ; END OF SYMCALC LOOP
	;---------------------------------------





	;----------------------------------------
	; variables for calls to FFTPACK5 
	; Space-time spectra is calculated that both 
	; positive and negative wave numbers are returned
	; while frequency is positive
	;               ..........................

	L      = lsector  ; length of time section
	M      = nlon     ; length of zonal section
	KDIM   = L/2+1
	LENSAV = L+4+floattointeger(log(L)) + 2*M+4+floattointeger(log(M))
	SD     = new((/M,KDIM/),float)   ; spectral density (freq,wave)   
	SD     = 0.0
	IER    = -999
	;----------------------------------------




	;---------------------------------------
	; FFT Initialization
	;               ..........................
	
	print("Initialize FFT routine")
  	wsave=new(LENSAV,"float")
  	FFT2D::spectra2di (L, M, KDIM, LENSAV, IER,wsave)
	;---------------------------------------




	;---------------------------------------
	; Calculate values for co-ordinate variables
	;               ..........................

	;
	; calculate wave numbers
	;---------------------------------------
	print("Make co-ordinate variables")
	wave = ispan( -(M/2), M/2-1, 1)
	wave!0 = "wave"
	wave&wave = wave
	wave@long_name = "zonal wavenumber"
	printVarSummary(wave)
	printVarSummary(SD)
	;
	; calculate frequencies
	;---------------------------------------
	freq   = fspan(0,0.5,KDIM)
	freq!0 = "freq"
	freq&freq      = freq
	freq@long_name = "frequency"
	freq@units     = freq_units
	;
        add_attributes(SD,(/"wave","freq"/))
        SD&wave = wave
        SD&freq = freq
        ;
        Nwav=dimsizes(wave({wv1:wv2}))
        Nfrq=dimsizes(freq({fr1:fr2}))

	spcf      = new ((/nlat,M,KDIM/),float)
	add_attributes(spcf,(/"lat","wave","freq"/))
	spcf&lat  = lat
        spcf&wave = wave
        spcf&freq = freq
	spcf      = 0.0
	;---------------------------------------



	;---------------------------------------
	; Main Loop of FFT calculations
	;               ..........................
	print("Going into FFT routine")
	do itim = 0,nrepeat-1
	print("time loop : "+(itim+1))
	nstart=lsector*(itim)
	nend=nstart+lsector-1
	;
	;print("Prepare the data for analysis")
	;print("Normalize, Detrend and Taper in time")
	;
	t_dt = varx(lat|:,lon|:,time|nstart:nend)
	t_dt = t_dt/stddev(t_dt)
	t_dt  = dtrend(t_dt,False)
	t_dt = taper(t_dt,0.2,0)   ; (lat,lon,time)
	;
	;print("Detrend and Taper in longitude")
	;
	t_dtx=t_dt(time|:,lat|:,lon|:)
	t_dtx=dtrend(t_dtx,False)
	t_dtx=taper(t_dtx,0.2,0)
	tp_dt = t_dtx(lat|:,lon|:,time|:)
	;
	; do the forward FFT for each latitude
	;
	do nl = 0, nlat-1
  	tmp = tp_dt(lon|:,time|:,lat|nl:nl)  
  	FFT2D::spectra2df (L, M, KDIM, LENSAV, tmp(:,:,0), SD, IER,wsave)
  	if (IER .ne. 0) then
    	print ("RFFT2DF: IER = "+IER)
    	exit
  	end if
        ; swap wave numbers for plotting
        ptmp                    = SD
        SD(0:M/2-2,:)           = (/ ptmp(M/2+1:M-1,:) /)
        SD(M/2-1:M-1,:)         = (/ ptmp(0:M/2,:) /)
        ;
        spcf(nl,:,:) = spcf(nl,:,:) + \
                (/ SD /)  /  (nrepeat*1.0)
	;
	end do  ; END of LOOP OVER LATITUDES
	;
	end do  ; END of LOOP OVER TIME
	print("Out of FFT routine")
	;---------------------------------------





	;---------------------------------------
	; PROCESS THE DATA FOR OUTPUT
	;               ..........................

	; Smooth the spectrum
	print("Smooth spectral estimates using a 3-point Daniell_Window")
        smooth=3
        dan_win=Daniell_Window(smooth)
        sclfactor=2.0/sum(dan_win^2)
	;
	; First in frequency
	;
	power1=wgt_runave_Wrap(spcf,dan_win,0)
	;
	; Next in wave number
	power2=wgt_runave_Wrap(power1(lat|:,freq|:,wave|:),dan_win,0)
	;
	;---------------------------------------




	;---------------------------------------
	; OUTPUT SECTION
	;               ..........................

	print("Output")
	nwv=dimsizes(wave)
	nfr=dimsizes(freq)
	fo = addfile(outfile,"c")
	;
	if (.not.LATAVG) then
	  print("Writing out all latitudes")
	  fo->fft_olr = power2(:,1:nfr-2,1:nwv-2)
	  exit
	end if
	;
	if (SYMCALC) then
	  print("Writing out values at 2 latitude points")
	  print("Southern point holds the Symmetric component")
	  print("Northern point holds the Asymmetric component")
	  fo->fft_sym = dim_avg_Wrap(power2(wave|1:nwv-2,freq|1:nfr-2,lat|0:lat1-1))
	  fo->fft_asym= dim_avg_Wrap(power2(wave|1:nwv-2,freq|1:nfr-2,lat|lat1:lat2))
	else
	  power=dim_avg_Wrap(power2(wave|1:nwv-2,freq|1:nfr-2,lat|:))
	  fo->fft_olr =power(freq|:,wave|:)
	end if
	;---------------------------------------

end
