/*
 *  vintrt_dump.c
 *
 *  $Id: vintrt_dump.c,v 1.4 2004/02/04 20:04:14 gnowicki Exp gnowicki $
 *
 *  utility to read vintrt packed pixel level data
 *
 *  $Log: vintrt_dump.c,v $
 *  Revision 1.4  2004/02/04 20:04:14  gnowicki
 *  Multiple files allowed on command line now.
 *
 *  Revision 1.3  2003/10/22 18:03:19  gnowicki
 *  Mostly comment and printf changes.
 *
 *
*/
#include <string.h>
#include "vintrt.h"

#define DEBUG 0
#define MAX_VARIABLE 20
#define FORMAT_LEN 80

static char RCS_ID[] = "$Id: vintrt_dump.c,v 1.4 2004/02/04 20:04:14 gnowicki Exp gnowicki $";

/*
 *
 */

void usage(char *);
int print_header(char *);

int main(int argc, char *argv[])
{
  const char ename[] = "\nvintrt_dump error:";
  const char dname[] = "\nvintrt_dump debug:";
  char *ifn;
  char *prog;
  extern char *optarg;
  extern int optind;
  int c, i, j;
  int nparams, nlines, npixels, dims[2];
  char var_name[MAX_VAR_NAME_SIZE + 1], units[MAX_UNIT_NAME_SIZE + 1];
  char form[FORMAT_LEN];
  int num_fill;
  float range[2], fill[MAX_FILL_VALUES], *data, *p;

/*
 *  program switches
 */
  int print_vnames = 0, print_ascii = 0, show_var = 0, show_data = 0;

  prog = argv[0];

  while ((c = getopt(argc, argv, "v:n:ahV")) != -1) {
    switch (c) {
      case 'a':
        print_ascii = 1;
        break;
      case 'h':
        print_vnames = 1;
        break;
      case 'n':
        show_var = 1;
        strncpy(var_name, optarg, MAX_VAR_NAME_SIZE + 1);
        break;
      case 'v':
/*
 *  if we're showing the data, display the variable parameters also.
 */
        show_data = 1;
        show_var = 1;
        strncpy(var_name, optarg, MAX_VAR_NAME_SIZE + 1);
        break;
      case 'V':
        printf("%s\n", RCS_ID);
        vintrt_utils_version();
        exit (0);
        break;
      default:
        usage(prog);
        break;
    }
  }

  if (optind >= argc) {
    usage(prog);
  }
  while (ifn = argv[optind++]) {
    if (vintrt_open(ifn, &nparams, &nlines, &npixels) == -1) {
      fprintf(stderr, "%s file %s cannot be opened correctly\n\n", ename, ifn);
      exit(-1);
    }

/*
 *  just printout the ascii file header
 */
    if (print_ascii == 1) {
      if (print_header(ifn) == -1) {
        fprintf(stderr, "%s print ascii header failedon filename: %s\n", ename, ifn);
        exit(-1);
      }
    }
/*
 *  printout the ascii file header and all of the information for the
 *  all of the variables
 */
    if (print_vnames == 1) {
      if (print_header(ifn) == -1) {
        fprintf(stderr, "%s print ascii header failedon filename: %s\n", ename, ifn);
        exit(-1);
      }
/*
 *  create a variable format string.
 */
      snprintf(form, FORMAT_LEN, "%%%ds  %%%ds  min: %%9.2f  max: %%9.2f  fill:",
               MAX_VAR_NAME_SIZE, MAX_UNIT_NAME_SIZE);
      for (i = 0; i < nparams; i++) {
        if (vintrt_get_vname(&i, var_name) != -1) {
          if (vintrt_var_attributes(var_name, units, range, fill, &num_fill) != -1) {
            printf(form, var_name, units, range[0], range[1]);
            for (j = 0; j < num_fill; j++) {
              printf(" %7.2f", fill[j]);
            }
            printf("\n");
          }
        }
      }
    }
/*
 *  printout the parameters of the requested variable.
 */
    if (show_var == 1) {
      if (vintrt_var_attributes(var_name, units, range, fill, &num_fill) == -1) {
        fprintf(stderr, "%s retrieve variable attributes failed on filename: %s\n", ename, ifn);
        exit (-1);
      }
/*
 *  create a variable format string.
 */
      snprintf(form, FORMAT_LEN, "%%%ds  %%%ds  min: %%9.2f  max: %%9.2f  fill:",
               MAX_VAR_NAME_SIZE, MAX_UNIT_NAME_SIZE);
      printf(form, var_name, units, range[0], range[1]);
      for (j = 0; j < num_fill; j++) {
        printf(" %7.2f", fill[j]);
      }
      printf("\n");
    }
/*
 *  printout the data of the requested variable.
 */
    if (show_data == 1) {
      if ((data = calloc(nlines * npixels, sizeof(float))) == NULL) {
        fprintf(stderr, "%s calloc data failed on filename: %s\n", ename, ifn);
        exit (-1);
      }
      dims[0] = nlines;
      dims[1] = npixels;
      if (vintrt_read_var(var_name, dims, data) == -1) {
        fprintf(stderr, "%s vintrt_read_data failed on filename: %s\n", ename, ifn);
        exit (-1);
      }
      for (i = 0, p = data; i < nlines; i++) {
        for (j = 0; j < npixels; j++, p++) {
          printf("%f ", *p);
        }
        printf("\n");
      }
      free(data);
    }

#if DEBUG
  printf("%s dims: %d %d %d\n", dname, nparams, nlines, npixels);
#endif
    vintrt_close();
  }
  exit (0);
}

/*
 *  print out program usage information and exit without error.
 */
void usage(char *s)
{
  printf("\nusage: %s [-a] [-h] [-V] [[-n|-v] <var name>] <filename [filename ...]>\n\n", s);

  printf("\tWhere:\n\n");
  printf("\t-a prints out header information\n");
  printf("\t-h prints out header information and the variables along with their properties\n");
  printf("\t-n <variable name> prints out the variable along with its properties\n");
  printf("\t-v <variable name> prints out the values for that particular variable\n");
  printf("\t\tname in one continous stream (a LOT of them) with newlines after each line.\n");
  printf("\t\tPrinted out line 1, pixel 1, line 1, pixel 2, etc.\n");
  printf("\t\tSingle spaces separate values on each line.\n");
  printf("\t-V prints out program version number\n\n");
  printf("\tThere are no restrictions on combining switches (except -V)!!\n\n");

  exit (0);
}

/*
 *  call the vintrt_utils routine vintrt_get_header() and then printout
 *  the ascii header returned to it.
 *
 *  return -1 on failure, 0 upon success.
 */
int print_header(char *fn)
{
  int m;
  char header[MAX_ASCII_SIZE + 1], form[FORMAT_LEN];

  if (vintrt_get_header(header) == -1) {
    return -1;
  }
  m = MAX_ASCII_SIZE < strlen(header) ? MAX_ASCII_SIZE : strlen(header);
/*
 *  create a variable format string.
 */
  snprintf(form, FORMAT_LEN, "%%%ds\n", m);
  printf("%s:\n", fn);
  printf(form, header);
  return 0;
}
