function color_echo
{
    local echo_options=
    local code_head=
    local code_tail=
    local message=
    while getopts "buc:n" option; do
        case $option in
        b)
            code_head=$code_head"\033[1m"
            code_tail="\033[0m"$code_tail
            ;;
        u)
            code_head=$code_head"\033[4m"
            code_tail="\033[0m"$code_tail
            ;;
        c)
            # For color code see http://tldp.org/LDP/abs/html/colorizing.html
            case $OPTARG in
            red)
                code_head=$code_head"\033[31;40m"
                code_tail="\033[0m"$code_tail
                ;;
            green)
                code_head=$code_head"\033[32;40m"
                code_tail="\033[0m"$code_tail
                ;;
            yellow)
                code_head=$code_head"\033[33;40m"
                code_tail="\033[0m"$code_tail
                ;;
            blue)
                code_head=$code_head"\033[34;40m"
                code_tail="\033[0m"$code_tail
                ;;
            *)
                OPTIND=0 # This is buggy!
                report_error "color_echo" "Unknown color."
            esac
            ;;
        n)
            echo_options="$echo_options -n"
            ;;
        ?)
            OPTIND=0 # This is buggy!
            report_error "color_echo" "Unknown option."
            ;;
        esac
    done
    eval message=\$$OPTIND # Indirect dereference, see http://tldp.org/LDP/abs/html/ivr.html
    eval echo $echo_options -e '$code_head$message$code_tail'
    OPTIND=0 # This is buggy! "getopts" does not reset OPTIND for you!
}

function report_error
{
    local reporter=$1
    local error=$2
    color_echo -b -c red -n "[Error]: "
    color_echo -u -c yellow -n "$reporter"
    echo ": $error"
    exit 1
}

function notice
{
    local reporter=$1
    local message=$2
    color_echo -b -c green -n "[Notice]: "
    color_echo -u -c yellow -n "$reporter"
    echo ": $message"
}

function check_file_exist
{
    local file=$1
    if [ ! -f $file ]; then
        report_error "$script_name: check_file_exist" \
            "File $file does not exist."
        exit 2
    fi
}

function check_dir_exist
{
    local dir=$1
    if [ ! -d $dir ]; then
        report_error "$script_name: check_dir_exist" \
            "Directory $dir does not exist."
        exit 2
    fi
}

function prepare_file
{
    local file=$1
    local override=$2
    if [ -f $file ]; then
        if [ $override == "no" ]; then
            report_error "$script_name" "File $file exists."
        elif [ $override == "yes" ]; then
            rm $file
        else
            report_error "$script_name" "Unknown whether to override or not."
        fi
    fi
    touch $file
    notice "$script_name" "File $file is generated."
}

function prepare_dir
{
    local dir=$1
    local override=$2
    if [ -d $dir ]; then
        if [ $override == "no" ]; then
            report_error "$script_name" "Directory $dir exists."
        elif [ $override == "yes" ]; then
            rm -r $dir
        else
            report_error "$script_name" "Unknown whether to override or not."
        fi
    fi
    mkdir $dir
    notice "$script_name" "Directory $dir is generated."
}

function get_suffix
{
    local file=$1
    echo $file | sed 's/^.*\.\(.\)/\1/g'
}

function look_for_command
{
    local cmd=$1
    which $cmd > /dev/null
    if [ $? != 0 ]; then
        report_error "$script_name" "Command $cmd is not found."
    fi
}

function create_cache_file
{
    local this="create_cache_file"
    local cache_file=$1
    if [ -f "$cache_file" ]; then
        report_error "$script_name: $this" "Cache file $cache_file exists."
    fi
    touch $cache_file
}

function write_cache_file
{
    local this="write_cache_file"
    local cache_file=$1
    local key=$2
    local value=$3
    if [ ! -f "$cache_file" ]; then
        report_error "$script_name: $this" "Cache file $cache_file is not found."
    fi
    awk -F ':' -v key="$key" -v value="$value" '
        BEGIN {tag = 0}
        {
            if ($1 == key) {
                printf ("%s: %s\n", key, value);
                tag = 1;
            } else {
                print;
            }
        }
        END {if (tag == 0) printf ("%s: %s\n", key, value)}' $cache_file > tmp
    cat tmp > $cache_file
    rm tmp
}

function read_cache_file
{
    local this="read_cache_file"
    local cache_file=$1
    local key=$2
    if [ ! -f "$cache_file" ]; then
        report_error "$script_name: $this" "Cache file $cache_file is not found."
    fi
    local value=$(awk -F ':' -v key="$key" '{if ($1 == key) print $2}' $cache_file)
    echo $value
}

function to_yes_or_no
{
    local this="to_yes_or_no"
    local true_or_false=$1
    if [ $true_or_false == "True" ]; then
        echo "y"
    elif [ $true_or_false == "False" ]; then
        echo "n"
    else
        report_error "$script_name: $this" "Argument \"$true_or_false\" is not correct"
    fi
}

function query_yes_or_no
{
    local this="query_yes_or_no"
    local question=$1
    local default=$2
    if [ -z "$default" ]; then
        default="True"
    fi
    echo "$question (y/n)" 1>&2
    read -p "[Default: $(to_yes_or_no $default)] > " ans
    if [ -n "$ans" ]; then
        if [ $ans == "y" ]; then
            echo "True"
        else
            echo "False"
        fi
    else
        echo $default
    fi
}

function query_string
{
    local this="query_string"
    local question=$1
    local default=$2
    echo "$question" 1>&2
    local ans=
    read -e -p "[Default: $default] > " ans
    if [ -n "$ans" ]; then
        echo $ans
    else
        echo $default
    fi
}

