#!/bin/bash

source bash_utils.sh

script_name=$(basename $0)
cache_file="show_parcels.cache"

function print_usage
{
    cat <<EOF
********************************************************************************
$script_name
Description:
    A BASH script plotting parcels output by ITS.
    It can be run into two mode, i.e., interactive or preset mode.
--------------------------------------------------------------------------------
Usage:
    -h - print this help;
    -c <file> - cache file for parsing settings in preset mode, the default is
                interactive mode.
********************************************************************************
EOF
}

function output_cache
{
    write_cache_file "$cache_file" "filewildcard" "$filewildcard"
    write_cache_file "$cache_file" "isPlotParcel" "$isPlotParcel"
    write_cache_file "$cache_file" "isPlotParcelOutline" "$isPlotParcelOutline"
    write_cache_file "$cache_file" "isPlotParcelFill" "$isPlotParcelFill"
    write_cache_file "$cache_file" "isPlotVertex" "$isPlotVertex"
    write_cache_file "$cache_file" "colormap" "$colormap"
    write_cache_file "$cache_file" "colorStride" "$colorStride"
    write_cache_file "$cache_file" "level1" "$level1"
    write_cache_file "$cache_file" "level2" "$level2"
    write_cache_file "$cache_file" "mapProj" "$mapProj"
    write_cache_file "$cache_file" "lonCnt" "$lonCnt"
    write_cache_file "$cache_file" "latCnt" "$latCnt"
    write_cache_file "$cache_file" "minLat" "$minLat"
    write_cache_file "$cache_file" "maxLat" "$maxLat"
    write_cache_file "$cache_file" "angle" "$angle"
    write_cache_file "$cache_file" "figtype" "$figtype"
    notice "$script_name" "Cache file $cache_file is generated."
}

function input_cache
{
    filewildcard=$(read_cache_file "$cache_file" "filewildcard")
    isPlotParcel=$(read_cache_file "$cache_file" "isPlotParcel")
    isPlotParcelOutline=$(read_cache_file "$cache_file" "isPlotParcelOutline")
    isPlotParcelFill=$(read_cache_file "$cache_file" "isPlotParcelFill")
    isPlotVertex=$(read_cache_file "$cache_file" "isPlotVertex")
    colormap=$(read_cache_file "$cache_file" "colormap")
    colorStride=$(read_cache_file "$cache_file" "colorStride")
    level1=$(read_cache_file "$cache_file" "level1")
    level2=$(read_cache_file "$cache_file" "level2")
    mapProj=$(read_cache_file "$cache_file" "mapProj")
    lonCnt=$(read_cache_file "$cache_file" "lonCnt")
    latCnt=$(read_cache_file "$cache_file" "latCnt")
    minLat=$(read_cache_file "$cache_file" "minLat")
    maxLat=$(read_cache_file "$cache_file" "maxLat")
    angle=$(read_cache_file "$cache_file" "angle")
    figtype=$(read_cache_file "$cache_file" "figtype")
    notice "$script_name" "Cache file $cache_file is parsed."
}

function debug
{
    echo $filewildcard
    echo $isPlotParcel
    echo $isPlotParcelOutline
    echo $isPlotParcelFill
    echo $isPlotVertex
    echo $colormap
    echo $colorStride
    echo $level1
    echo $level2
    echo $mapProj
    echo $lonCnt
    echo $latCnt
    echo $minLat
    echo $maxLat
    echo $angle
    echo $figtype

}

look_for_command ncl

read_from_file=no
while getopts "hc:" option; do
    case $option in
    h)
        print_usage
        exit 0
        ;;
    c)
        read_from_file=yes
        cache_file=$OPTARG
        ;;
    ?)
        print_usage
        exit 1
        ;;
    esac
done
OPTIND=0

if [ $read_from_file == "no" ]; then
    if [ -f "$cache_file" ]; then
        # If the cache file exists, read it in
        input_cache
    else
        # Default values
        create_cache_file "$cache_file"
        filewildcard="no default"
        isPlotParcel="True"
        isPlotParcelOutline="True"
        isPlotParcelFill="True"
        isPlotVertex="True"
        colormap="WhBlGrYeRe"
        colorStride="1"
        level1=0.0
        level2=0.0
        mapProj="CE"
        lonCnt=0
        latCnt=0
        minLat=80.0
        maxLat=-80.0
        angle=1.0
        figtype="x11"
    fi

    filewildcard=$(query_string "Input the data file wildcard:" "$filewildcard")
    isPlotParcel=$(query_yes_or_no "Plot parcels?" $isPlotParcel)

    if [ $isPlotParcel == "True" ]; then
        isPlotParcelOutline=$(query_yes_or_no "Outline parcels?" $isPlotParcelOutline)
        if [ $isPlotParcelOutline == "False" ]; then
            echo "** Color fill of parcels is used."
            isPlotParcelFill="True"
        else
            isPlotParcelFill=$(query_yes_or_no "Fill parcels with color?" $isPlotParcelFill)
        fi
        if [ $isPlotParcelFill == "True" ]; then
            echo "Input the level range:"
            read -p "[Default $level1~$level2] > " ans1 ans2
            if [ -n "$ans1" ]; then
                level1=$ans1
                level2=$ans2
            fi
            colormap=$(query_string "Input color map (see NCL website):" "$colormap")
            colorStride=$(query_string "Input color stride for color bar:" "$colorStride")
        fi
    else
        isPlotParcelOutline="False"
        isPlotParcelFill="False"
    fi

    isPlotVertex=$(query_yes_or_no "Plot vertices?" $isPlotVertex)

    mapProj=$(query_string "Input map projection (NH/SH/CE/ST):" "$mapProj")

    if [ $mapProj == "CE" ]; then
        echo "** Cylindrical equidistant projection is used."
    fi

    if [ $mapProj == "NH" ]; then
        echo "** Northen stereographic projection is used."
        minLat=$(query_string "Input the southest latitude to view:" "$minLat")
    fi

    if [ $mapProj == "SH" ]; then
        echo "** Southen stereographic projection is used."
        maxLat=$(query_string "Input the northest latitude to view:" "$maxLat")
    fi

    if [ $mapProj == "ST" ]; then
        echo "** Satellite projection is used."
        echo "Input the center of viewport:"
        read -p "[Default: lon=$lonCnt, lat=$latCnt] > " ans1 ans2
        if [ -n "$ans1" -a -n "$ans2" ]; then
            lonCnt=$ans1
            latCnt=$ans2
        fi
        angle=$(query_string "Input the angle of viewport:" "$angle")
    fi

    figtype=$(query_string "Input figure type: (pdf/x11)" "$figtype")

    output_cache
else
    input_cache
fi

ncl <<-EOF
load "$NCARG_ROOT/lib/ncarg/nclscripts/csm/gsn_code.ncl"
load "$NCARG_ROOT/lib/ncarg/nclscripts/csm/gsn_csm.ncl"
load "$NCARG_ROOT/lib/ncarg/nclscripts/csm/contributed.ncl"

begin

    ; *************************************************************************
    ; Conversion parameter
    Rad2Deg = 45./atan(1.)

    ; *************************************************************************
    ; Read in data file names
    files = systemfunc("ls $filewildcard")

    ; *************************************************************************
    ; Some dimension information
    numTimeStep = dimsizes(files)

    ; *************************************************************************
    ; Open workstation
    figtype = "$figtype"
    if (figtype .eq. "pdf") then
        figtype@wkOrientation = "Landscape"
    end if
    wks = gsn_open_wks(figtype, "parcel")

    ; *************************************************************************
    ; Set resources for map
    mapRes = True
    mapRes@gsnFrame = False
    mapRes@gsnMaximize = True
    mapRes@mpFillOn = False
    mapRes@mpOutlineOn = False
    mapRes@mpGreatCircleLinesOn = True
    mapRes@mpGridLatSpacingF = 1.5
    mapRes@mpGridLineColor = "background"
    mapRes@mpCenterLonF = 180

    if ("$mapProj" .eq. "ST") then
        mapRes@mpProjection = "satellite"
        mapRes@mpCenterLonF = $lonCnt
        mapRes@mpCenterLatF = $latCnt
        mapRes@mpLimitMode = "angles"
        mapRes@mpLeftAngleF = $angle
        mapRes@mpRightAngleF = $angle
        mapRes@mpBottomAngleF = $angle
        mapRes@mpTopAngleF = $angle
    end if
    if ("$mapProj" .eq. "SH") then
        mapRes@mpProjection = "stereographic"
        mapRes@gsnPolar = "SH"
        mapRes@mpMaxLatF = $maxLat
    end if
    if ("$mapProj" .eq. "NH") then
        mapRes@mpProjection = "Stereographic"
        mapRes@gsnPolar = "NH"
        mapRes@mpMinLatF = $minLat
    end if

    ; *************************************************************************
    ; Define colors and set color bar
    if ($isPlotParcel .and. $isPlotParcelFill) then
        gsn_define_colormap(wks, "$colormap")
        cmap = gsn_retrieve_colormap(wks)
        numLevel = dimsizes(cmap(:,0))-3
        stride = $colorStride
        numColor = numLevel/stride
        levels = new(numLevel, float)
        levels = fspan($level1, $level2, numLevel)
        colors = new((/numColor,3/), float)
        colorLabels = new(numColor, string)
        do i = 0, numColor-1
            colors(i,:) = cmap(2+i*stride,:)
            colorLabels(i) = sprintf("%5.2f", levels(i*stride))
        end do
        lbRes = True
        lbRes@lbOrientation = "Horizontal"
        lbRes@vpHeightF = 0.1
        lbRes@vpWidthF = 0.7
        lbRes@lbLabelAngleF = 45.0
        lbRes@lbLabelFontHeightF = 0.015
        lbRes@lbAutoManage = False
        lbRes@lbFillColors = colors
        lbRes@lbPerimOn = False
        lbRes@lbMonoFillPattern = True
        lbRes@lbLabelAutoStride = False
        lbRes@lbLabelStride = 6
        gsn_labelbar_ndc(wks, numColor, colorLabels, 0.13, 0.23, lbRes)
    end if

    ; *************************************************************************
    ; Set resources for parcel edges and colors
    edgeRes = True
    edgeRes@gsLineThicknessF = 0.5
    edgeRes@gsLineColor = "blue"
    fillRes = True

    ; *************************************************************************
    ; Report quantities
    minDensity = 999.0d0
    maxDensity = -999.0d0

    do i = 0, numTimeStep-1
        f = addfile(files(i), "r")

        if (isfilevar(f, "density")) then
            minDensity = min(f->density)
            maxDensity = max(f->density)
            mapRes@gsnLeftString = sprintf("min: %8.6f", minDensity)+", "+ \
                                   sprintf("max: %8.6f", maxDensity)
        end if

        map = gsn_csm_map(wks, mapRes)
        dims = getfiledimsizes(f)
        numParcel = dims(0)
        if ($isPlotParcel .and. isfilevar(f, "lonVtx")) then
            n = 0
            do k = 0, numParcel-1
                numVtx = f->idxVtx(n)
                n = n+1
                lon = new(numVtx+1, "double")
                lat = new(numVtx+1, "double")
                do j = 0, numVtx-1
                    m = f->idxVtx(n)-1
                    n = n+1
                    lon(j) = f->lonVtx(m)*Rad2Deg
                    lat(j) = f->latVtx(m)*Rad2Deg
                end do
                lon(numVtx) = lon(0)
                lat(numVtx) = lat(0)
                if ($isPlotParcelOutline) then
                    gsn_polyline(wks, map, lon, lat, edgeRes)
                end if
                if ($isPlotParcelFill) then
                    ;if (max(lon)-min(lon) .lt. 90) then
                        fillRes@gsFillColor = GetFillColor(levels, cmap, \
                            f->density(k))
                        gsn_polygon(wks, map, lon, lat, fillRes)
                    ;end if
                end if
                delete(lon)
                delete(lat)
            end do
        end if
        if ($isPlotVertex .and. isfilevar(f, "lonVtx")) then
            ; Plot vertices
            pntRes = True
            pntRes@gsMarkerIndex = 1
            pntRes@gsMarkerSizeF = 0.001
            pntRes@gsMarkerColor = "red"
            gsn_polymarker(wks, map, f->lonVtx*Rad2Deg, \
                f->latVtx*Rad2Deg, pntRes)
        end if
        frame(wks)
        system("echo time step "+i+ \
               " max value: "+maxDensity+\
               " min value: "+minDensity)
        delete(f)
    end do

end
EOF

